<?php

class Restapi
{
    const CHUNK_SIZE = 1024*1024;

    private $_username;

    private $_password;

    private $_checkAction;

    private $_url;

    private $_method;

    private $_date;

    private $_contentMd5;

    private $_async;

    /**
     * Restapi constructor.
     * @param string|null $username
     * @param string|null $password
     * @param int|null $action
     */
    public function __construct($username = null, $password = null, $action = null)
    {
        $this->_username = $username;
        $this->_password = $password;
        $this->_checkAction = $action;
    }

    /**
     * 设置用户信息
     * @param string $username
     * @param string $password
     * @return Restapi
     */
    public function setUserInfo($username, $password)
    {
        if(is_string($username) && is_string($password)){
            $this->_username = $username;
            $this->_password = $password;
        }
        return $this;
    }

    /**
     * 设置验证方式 0为基础认证，其他为签名认证
     * @param int $action
     * @return Restapi
     */
    public function setCheckAction($action)
    {
        if (is_numeric($action)) {
            $this->_checkAction = $action;
        }
        return $this;
    }

    /**
     * 设置请求的url
     * @param string $url
     * @return Restapi
     */
    public function setUrl($url)
    {
        if(is_string($url)){
            $urlParse = parse_url($url);
            $pathInfo = substr($urlParse['path'], 1);
            $queryString = $urlParse['query'];
            $this->_url = $urlParse['scheme'].'://'.$urlParse['host'].'/'.urlencode($pathInfo);
            if($queryString){
                $this->_url .= '?'.$queryString;
            }
        }
        return $this;
    }

    /**
     * 设置请求方法
     * @param string $method
     * @return Restapi
     */
    public function setMethod($method)
    {
        if(is_string($method)){
            $this->_method = $method;
        }
        return $this;
    }

    /**
     * 设置日期(基础认证可不设置)
     * @param string $date
     * @return Restapi
     */
    public function setDate($date)
    {
        if(is_string($date)){
            $this->_date = $date;
        }
        return $this;
    }

    /**
     * 设置文件内容的md5加密
     * @param $md5
     * @return $this
     */
    public function setContentMd5($md5){
        if(is_string($md5)){
            $this->_contentMd5 = $md5;
        }
        return $this;
    }

    /**
     * 设置是否异步执行，适用于删除和上传文件
     * @param $async
     * @return $this
     */
    public function setAsync($async){
        $this->_async = $async;
        return $this;
    }

    /**
     * 获取认证
     * @return string
     */
    public function getAuthorization()
    {

        if ($this->_checkAction === 0) {
            return 'Basic ' . base64_encode($this->_username . ":" . $this->_password);
        }

        $parse = parse_url($this->_url);

        $uri = $parse['path'];

        if(isset($parse['query'])){
            $uri .= "?".$parse['query'];
        }

        $this->_url = $parse['host'].$uri;

        $data = $this->_method . "&" . $uri . "&" . $this->_date;

        if($this->_contentMd5){
            $data .= '&' . $this->_contentMd5;
        }

        $password = base64_encode($this->_password);

        $sign = hash_hmac('sha1', $data, $password, true);

        return ' WESTYUN ' . $this->_username . ':' . base64_encode($sign);
    }

    /**
     * 获取目录数据
     * @param string|null $iter
     * @param int|null $limit
     * @param string|null $order
     * @return array|string
     * @throws \Exception
     */
    public function getList($iter = null, $limit = null, $order = null)
    {
        $authorization = $this->getAuthorization();
        $headers = [
            'Authorization' => $authorization,
        ];
        if($this->_date){
            $headers['Date'] = $this->_date;
        }
        if ($iter) {
            $headers['x-list-iter'] = $iter;
        }
        if ($limit) {
            $headers['x-list-limit'] = $limit;
        }
        if ($order) {
            $headers['x-list-order'] = $order;
        }
        $data = self::fssCurl($this->_url, $headers, $this->_method);
        return $data;
    }

    /**
     * 分块获取文件内容
     * @param int $start
     * @param int $end
     * @return string
     * @throws \Exception
     */
    public function getContent($start = null, $end = null)
    {
        $authorization = $this->getAuthorization();
        $headers = [
            'Authorization' => $authorization,
        ];
        if($this->_date){
            $headers['Date'] = $this->_date;
        }
        if ($start !== null && $end === null) {
            $headers['Range'] = "bytes=" . $start . "-";
        } elseif ($start === null && $end !== null) {
            $headers['Range'] = "bytes= -" . $end;
        }
        if ($start !== null && $end !== null) {
            $headers['Range'] = "bytes=" . $start . "-" . $end;
        }

        $data = self::fssCurl($this->_url, $headers, $this->_method);
        if (!isset($data['data'])) {
            throw new Exception("获取文件内容异常");
        }
        return $data['data'];
    }

    /**
     * 获取文件信息
     * @return array
     * @throws \Exception
     */
    public function getFileInfo()
    {
        $authorization = $this->getAuthorization();
        $headers = [
            'Authorization' => $authorization,
        ];
        if($this->_date){
            $headers['Date'] = $this->_date;
        }
        $data = self::fssCurl($this->_url, $headers, $this->_method);
        $info = [];
        foreach ($data as $k => $v){
            $tmp = explode(':',$v);
            if(preg_match('/x-west/',$tmp[0])){
                $info[$tmp[0]] = $tmp[1];
            }
        }
        return $info;
    }

    /**
     * 复制or移动文件
     * @param string $source
     * @param int $type 0-复制 others-移动
     * @param string $directive
     * @param array $meta 文件元信息
     * @return mixed
     * @throws \Exception
     */
    public function copyOrMoveFile($source, $type, $directive = null, $meta = null)
    {
        try {
            $authorization = $this->getAuthorization();
            $headers = [
                'Authorization' => $authorization,
            ];
            if($this->_date){
                $headers['Date'] = $this->_date;
            }
            if ($type == 0) {
                $headers['x-west-copy-source'] = $source;
            } else {
                $headers['x-west-move-source'] = $source;
            }
            if ($directive) {
                $headers['x-west-metadata-directive'] = $directive;
            }
            if($meta){
                foreach ($meta as $k => $v){
                    $key = 'x-west-meta-'.$k;
                    $headers[$key] = $v;
                }
            }
            return self::fssCurl($this->_url, $headers, $this->_method);
        } catch (\Exception $e) {
            return false;
        }
    }

    /**
     * 创建目录
     * @return mixed
     * @throws \Exception
     */
    public function createFolder()
    {
        try {
            $authorization = $this->getAuthorization();
            $headers = [
                'Authorization' => $authorization,
                'folder' => 'true'
            ];
            if($this->_date){
                $headers['Date'] = $this->_date;
            }
            return self::fssCurl($this->_url, $headers, $this->_method);
        } catch (\Exception $e) {
            return false;
        }
    }

    /**
     * 删除目录或文件
     * @return mixed
     * @throws \Exception
     */
    public function deleteFolderOrFlie()
    {
        try {
            $authorization = $this->getAuthorization();
            $headers = [
                'Authorization' => $authorization,
            ];
            if($this->_date){
                $headers['Date'] = $this->_date;
            }
            if($this->_async){
                $headers['x-west-async'] = 'true';
            }
            return self::fssCurl($this->_url, $headers, $this->_method);
        } catch (\Exception $e) {
            return false;
        }
    }

    /**
     * 设置文件的metadata
     * @param array $metaInfo
     * @return mixed
     * @throws \Exception
     */
    public function setMetadata(array $metaInfo)
    {
        try {
            $authorization = $this->getAuthorization();
            $headers = [
                'Authorization' => $authorization,
            ];
            if($this->_date){
                $headers['Date'] = $this->_date;
            }
            foreach ($metaInfo as $k => $v) {
                $key = 'x-west-meta-' . $k;
                $headers[$key] = $v;
            }
            return self::fssCurl($this->_url, $headers, $this->_method);
        } catch (\Exception $e) {
            return false;
        }
    }

    /**
     * 上传文件
     * @param string $path
     * @param string|null $type
     * @param string|null $secret
     * @param int|null $ttl
     * @param string|null $thumb
     * @param array|null $origin
     * @return mixed
     * @throws \Exception
     */
    public function uploadFile($path, $type = null, $secret = null, $ttl = null, $thumb = null, $origin = null)
    {
        try {
            $data = file_get_contents($path);
            $authorization = $this->getAuthorization();
            $headers = [
                'Authorization' => $authorization,
                'Content-Length' => strlen($data)
            ];
            if($this->_async){
                $headers['x-west-async'] = 'true';
            }
            if($this->_contentMd5){
                $headers['Content-MD5'] = $this->_contentMd5;
            }
            if($this->_date){
                $headers['Date'] = $this->_date;
            }
            if ($type) {
                $headers['Content-Type'] = $type;
            }
            if ($secret) {
                $headers['Content-Secret'] = $secret;
            }
            if ($ttl) {
                $headers['x-west-meta-ttl'] = $ttl;
            }
            if ($thumb) {
                $headers['x-gmkerl-thumb'] = $thumb;
            }
            if (!empty($origin)) {
                foreach ($origin as $k => $v) {
                    $key = 'x-west-meta-' . $k;
                    $headers[$key] = $v;
                }
            }
            return self::fssCurl($this->_url, $headers, $this->_method, $data);
        } catch (\Exception $e) {
            return false;
        }
    }

    /**
     * 分块上传文件
     * @param $path
     * @param string|null $type
     * @param string|null $secret
     * @param int|null $ttl
     * @param array|null $origin
     * @return bool
     */
    public function uploadBigFile($path, $type = null, $secret = null, $ttl = null, $origin = null){
        try{
            $length = filesize($path);
            $initResult = $this->bigFileInit($length,$type,$secret,$ttl,$origin);
            if($initResult === false){
                throw new Exception('upload init failed');
            }
            $fp = fopen($path, "r");
            $i = 0;
            while (!feof($fp)) {
                $buffer = fread($fp, self::CHUNK_SIZE);
                $uploadRes = $this->uploadChunk($i,$buffer);
                if($uploadRes === false){
                    throw new Exception('第'.$i.'块分片上传失败');
                }
                $i += 1;
            }
            fclose($fp);
            $completeRes = $this->bigFileComplete();
            if($completeRes === false){
                return false;
            }
            return true;
        }catch (Exception $e){
            return false;
        }
    }

    /**
     * 初始化大文件
     * @param int $length
     * @param string|null $type
     * @param string|null $secret
     * @param int|null $ttl
     * @param array|null $origin
     * @return bool
     * @throws Exception
     */
    private function bigFileInit($length, $type = null, $secret = null, $ttl = null, $origin = null){
        $authorization = $this->getAuthorization();
        $headers = [
            'Authorization' => $authorization,
            'x-west-multi-disorder' => 'true',
            'x-west-multi-stage' => 'initiate',
            'x-west-multi-length' => $length
        ];
        if($this->_date){
            $headers['Date'] = $this->_date;
        }
        if ($type) {
            $headers['x-west-multi-type'] = $type;
        }
        if ($secret) {
            $headers['Content-Secret'] = $secret;
        }
        if ($ttl) {
            $headers['x-west-meta-ttl'] = $ttl;
        }
        if (!empty($origin)) {
            foreach ($origin as $k => $v) {
                $key = 'x-west-meta-' . $k;
                $headers[$key] = $v;
            }
        }
        $res = $this->fssCurl($this->_url,$headers,$this->_method);
        if($res['code'] == 200){
            return true;
        }
        return false;
    }

    /**
     * 分块上传
     * @param int $chunkId
     * @param string $data
     * @return bool
     * @throws Exception
     */
    private function uploadChunk($chunkId,$data){
        $headers = [
            'x-west-multi-disorder' => 'true',
            'x-west-multi-stage' => 'upload',
            'x-west-part-id' => $chunkId
        ];
        $authorization = $this->getAuthorization();
        $headers['Authorization'] = $authorization;
        if($this->_date){
            $headers['Date'] = $this->_date;
        }
        $res = $this->fssCurl($this->_url,$headers,$this->_method,$data);
        if($res['code'] == 200){
            return true;
        }
        return false;
    }

    /**
     * 上传完成
     * @return bool
     * @throws Exception
     */
    private function bigFileComplete(){
        $headers = [
            'x-west-multi-disorder' => 'true',
            'x-west-multi-stage' => 'complete',
        ];
        $authorization = $this->getAuthorization();
        $headers['Authorization'] = $authorization;
        if($this->_date){
            $headers['Date'] = $this->_date;
        }
        $res = $this->fssCurl($this->_url,$headers,$this->_method);
        if($res['code'] == 200){
            return true;
        }
        return false;
    }


    /**
     * fss curl
     * @param string $url
     * @param array $headers
     * @param string $method
     * @param string $data
     * @return array
     * @throws \Exception
     */
    public function fssCurl($url, $headers, $method, $data = null)
    {

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_BINARYTRANSFER, true);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, $method);
        if ($data) {
            curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
        }
        if (substr($url, 0, 5) == 'https') {
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
            curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
        }
        if ($method === 'HEAD') {
            curl_setopt($ch, CURLOPT_HEADER, true);
            curl_setopt($ch, CURLOPT_NOBODY, true);
        }
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        if (!empty($headers)) {
            $headarr = [];
            foreach ($headers as $k => $v) {
                $headarr[] = $k . ": " . $v;
            }
            curl_setopt($ch, CURLOPT_HTTPHEADER, $headarr);
        }
        $res = curl_exec($ch);
        $response = json_decode($res, true);
        curl_close($ch);
        if ($response === null) {
            if ($method === 'HEAD') {
                $res = explode("\r\n", $res);
                return $res;
            }
            return ['data' => $res];
        }

        return $response;
    }

}